unit Competences;

interface

uses
  SysUtils, Classes, Graphics
  ;

type
  TCompetence = class
    Nom : string;
    Couleur : TColor;
    Valorisation : integer;
  public
    constructor Create(const aNom : string; const aCouleur : TColor = clWhite); reintroduce; virtual;
  public
    procedure ChangeValorisation(const aValeur: integer); virtual;
    procedure Ameliore(const aValeur : integer); virtual;
    procedure Regresse(const aValeur : integer); virtual;
  end;

type
  TListeCompetences = class(TStringList)
  public
    constructor Create(); virtual;
    destructor Destroy(); override;
  public
    function Ajoute(const aNom : string; const aCouleur : TColor = clWhite) : integer; overload; virtual;
    function Ajoute(const aCompetence : TCompetence) : integer; overload; virtual;
    function DonneCompetence(const aNum : integer) : TCompetence; overload; virtual;
    function DonneCompetence(const aNom : string) : TCompetence; overload; virtual;
    function DonneValorisation(const aNomCompetence : string = '') : integer; virtual;
  end;

type
  TEquipe = class
    Nom : string;
    Nombre : integer;
    lstCompetencesSucces : TListeCompetences;
    lstCompetencesEchec : TListeCompetences;
  public
    constructor Create(const aNom : string); reintroduce; overload; virtual;
    destructor Destroy(); override;
  public
    function GetNombre() : integer; virtual;
    function GetScore() : integer; virtual;
  public
    property Nb : integer read GetNombre write Nombre;
    property Score : integer read GetScore;
  end;

type
  TPhaseApprentissage = class
    Nom : string;
    Valorisation : integer;
    EchecPenalise : boolean;
    NbItems : integer;
    NomFichierVideo : string;
  public
    constructor Create(const aNom : string; const aValeur : integer; const aEchecPenalise : boolean; const aNbItems : integer); reintroduce; overload; virtual;
    constructor Create(const aSectionIni : TStrings; const aNomPhase : string); reintroduce; overload; virtual;
  end;

type
  TSessionPedagogique = class
    Nom : string;
    lstPhases : TStringList;
    lstCompetences : TListeCompetences;
    lstEquipes : TStringList;
  public
    constructor Create(const aNom : string); reintroduce; overload;
    destructor Destroy(); override;
  public
    function Ajoute(const aPhase : TPhaseApprentissage) : integer; virtual;
    function DonnePhase(const aNum : integer) : TPhaseApprentissage; overload; virtual;
    function DonnePhase(const aNom : string) : TPhaseApprentissage; overload; virtual;
    function DonneEquipe(const aNum : integer) : TEquipe; overload; virtual;
    function DonneEquipe(const aNom : string) : TEquipe; overload; virtual;
  end;

type
  TFaceFace = class(TSessionPedagogique)
  public
    constructor Create(const aNom : string); overload; virtual;
  public
    function DonneEquipe1() : TEquipe; virtual;
    function DonneEquipe2() : TEquipe; virtual;
  public
    property Equipe1 : TEquipe read DonneEquipe1;
    property Equipe2 : TEquipe read DonneEquipe2;
  end;

type
  TBurgerCriz = class(TFaceFace)
  public
    constructor Create(); overload; virtual;
  public
    property EquipeKetchup : TEquipe read DonneEquipe1;
    property EquipeMayo : TEquipe read DonneEquipe2;
  end;

implementation

uses Math;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TCompetence }

constructor TCompetence.Create(const aNom : string; const aCouleur : TColor);
begin
  inherited Create();
  Nom := aNom;
  Couleur := aCouleur;
  Valorisation := 0;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

procedure TCompetence.ChangeValorisation(const aValeur: integer);
begin
  Valorisation := Valorisation + aValeur;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

procedure TCompetence.Ameliore(const aValeur: integer);
begin
  ChangeValorisation(Abs(aValeur));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

procedure TCompetence.Regresse(const aValeur: integer);
begin
  ChangeValorisation((-1) * Abs(aValeur));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TListeCompetences }

constructor TListeCompetences.Create();
begin
  inherited;
  Duplicates := dupError;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

destructor TListeCompetences.Destroy();
var
    kNumCompetence : integer;
begin
  for kNumCompetence := 0 to (Count - 1) do
  begin
    Objects[kNumCompetence].Free();
  end;
  inherited;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TListeCompetences.Ajoute(const aNom : string; const aCouleur : TColor) : integer;
begin
  Result := Ajoute(TCompetence.Create(aNom, aCouleur));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TListeCompetences.Ajoute(
  const aCompetence: TCompetence
  ): integer;
begin
  Result := AddObject(aCompetence.Nom, aCompetence);
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TListeCompetences.DonneCompetence(const aNum : integer) : TCompetence;
begin
  Result := nil;
  if ((aNum > -1) and (aNum < count)) then
  begin
    Result := TCompetence(Objects[aNum]);
  end;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TListeCompetences.DonneCompetence(const aNom : string) : TCompetence;
begin
  Result := DonneCompetence(IndexOf(aNom));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TListeCompetences.DonneValorisation(
  const aNomCompetence : string
  ) : integer;
var
    kNbOccurrences : integer;
    kCompetence : TCompetence;
    kNumCompetence : integer;
begin
  kNbOccurrences := 0;
  kNumCompetence := 0;
  if (aNomCompetence <> '') then
  begin
    kCompetence := DonneCompetence(aNomCompetence);
  end else
  begin
    kCompetence := DonneCompetence(kNumCompetence);
  end;
  while (Assigned(kCompetence)) do
  begin
    kNbOccurrences := kNbOccurrences + kCompetence.Valorisation;
    if (aNomCompetence <> '') then
    begin
      kCompetence := nil;
    end else
    begin
      Inc(kNumCompetence);
      kCompetence := DonneCompetence(kNumCompetence);
    end;
  end;
  Result := kNbOccurrences;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TEquipe }

constructor TEquipe.Create(const aNom: string);
begin
  inherited Create();
  Nom := aNom;
  lstCompetencesSucces := TListeCompetences.Create();
  lstCompetencesEchec := TListeCompetences.Create();
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

destructor TEquipe.Destroy();
begin
  lstCompetencesEchec.Clear();
  lstCompetencesEchec.Free();
  lstCompetencesSucces.Clear();
  lstCompetencesSucces.Free();
  inherited;
end;
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TEquipe.GetNombre() : integer;
begin
  Result := Nombre;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TEquipe.GetScore() : integer;
begin
  Result := lstCompetencesSucces.DonneValorisation('')
    - lstCompetencesEchec.DonneValorisation('')
    ;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TPhaseApprentissage }

constructor TPhaseApprentissage.Create(const aNom: string;
  const aValeur: integer; const aEchecPenalise: boolean;
  const aNbItems: integer);
begin
  Nom := aNom;
  Valorisation := aValeur;
  EchecPenalise := aEchecPenalise;
  NbItems := Min(1, aNbItems);
  NomFichierVideo := '';
end;

constructor TPhaseApprentissage.Create(const aSectionIni : TStrings; const aNomPhase : string);
var
    kNomCle : string;
    kIndexCleIni : integer;
begin
  Nom := aNomPhase;
  kNomCle := 'NbMiams';
  kIndexCleIni := aSectionIni.IndexOfName(kNomCle);
  if (kIndexCleIni <> -1) then
    Valorisation := StrToInt(aSectionIni.Values[kNomCle]);
  kNomCle := 'Cadeau';
  kIndexCleIni := aSectionIni.IndexOfName(kNomCle);
  if (kIndexCleIni <> -1) then
    EchecPenalise := (aSectionIni.Values[kNomCle] = 'Oui');
  kNomCle := 'Video';
  kIndexCleIni := aSectionIni.IndexOfName(kNomCle);
  if (kIndexCleIni <> -1) then
    NomFichierVideo := aSectionIni.Values[kNomCle];
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TSessionPedagogique }

constructor TSessionPedagogique.Create(const aNom: string);
begin
  inherited Create();
  Nom := aNom;
  lstPhases := TStringList.Create();
  lstCompetences := TListeCompetences.Create();;
  lstEquipes := TStringList.Create();
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

destructor TSessionPedagogique.Destroy();
begin
  lstEquipes.Clear();
  FreeAndNil(lstEquipes);
  lstCompetences.Clear();
  FreeAndNil(lstCompetences);
  lstPhases.Clear();
  FreeAndNil(lstPhases);
  inherited;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TSessionPedagogique.Ajoute(const aPhase: TPhaseApprentissage): integer;
begin
  Result := lstPhases.AddObject(aPhase.Nom, aPhase);
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TSessionPedagogique.DonnePhase(
  const aNum: integer): TPhaseApprentissage;
begin
  Result := nil;
  if ((aNum > -1) and (aNum < lstPhases.count)) then
  begin
    Result := TPhaseApprentissage(lstPhases.Objects[aNum]);
  end;
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TSessionPedagogique.DonnePhase(
  const aNom: string): TPhaseApprentissage;
begin
  Result := DonnePhase(lstPhases.IndexOf(aNom));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TSessionPedagogique.DonneEquipe(const aNum: integer): TEquipe;
begin
  Result := nil;
  if ((aNum > -1) and (aNum < lstEquipes.count)) then
    Result := TEquipe(lstEquipes.Objects[aNum]);
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TSessionPedagogique.DonneEquipe(const aNom: string): TEquipe;
begin
  Result := DonneEquipe(lstEquipes.IndexOf(aNom));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TFaceFace }

constructor TFaceFace.Create(const aNom: string);
begin
  inherited;
  lstEquipes.AddObject('Equipe1', TEquipe.Create('Equipe1'));
  lstEquipes.AddObject('Equipe2', TEquipe.Create('Equipe2'));
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TFaceFace.DonneEquipe1(): TEquipe;
begin
  Result := DonneEquipe(0);
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TFaceFace.DonneEquipe2() : TEquipe;
begin
  Result := DonneEquipe(1);
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

{ TBurgerCriz }

constructor TBurgerCriz.Create();
begin
  inherited Create('Burger Criz');
  EquipeKetchup.Nom := 'Ketchup';
  EquipeMayo.Nom := 'Mayo';
end;

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

end.
